// This is a script that parses Synchronet log files to generate door stats.
// Version 1.01

"use strict";

require("sbbsdefs.js", "K_NOCRLF");

var gLogsDir = backslash(system.logs_dir + "logs");
var gStatsJsonFilename = system.mods_dir + "DDDoorStats.json";

/*
var doorStats = compileAllDoorStats();
printConsoleOrNot("Door stats by door:");
for (var doorName in  doorStats.byDoor)
{
	printConsoleOrNot(doorName + ":");
	printConsoleOrNot(" # times ran: " + doorStats.byDoor[doorName].numTimesRan);
	printConsoleOrNot(" Last run login time: " + doorStats.byDoor[doorName].lastRunLoginDateStr);
	printConsoleOrNot(" Last run by: " + doorStats.byDoor[doorName].lastRunUsername);
	printConsoleOrNot(" Most common user: " + doorStats.byDoor[doorName].mostCommonUser);
	printConsoleOrNot(" User run counts:");
	for (var username in doorStats.byDoor[doorName].usernameRunCounts)
		printConsoleOrNot("  " + username + ": " + doorStats.byDoor[doorName].usernameRunCounts[username]);
}
printConsoleOrNot("");
printConsoleOrNot("Door stats by user:");
for (var username in doorStats.byUsername)
{
	printConsoleOrNot(username + ":");
	printConsoleOrNot(" Total # doors ran: " + doorStats.byUsername[username].totalNumDoorsRan);
	printConsoleOrNot(" Last login time: " + doorStats.byUsername[username].lastLoginDateStr);
	printConsoleOrNot(" Most common door: " + doorStats.byUsername[username].mostCommonDoor);
	printConsoleOrNot(" Door run counts:");
	for (var doorName in doorStats.byUsername[username].doorRunCounts)
		printConsoleOrNot("  " + doorName + ": " + doorStats.byUsername[username].doorRunCounts[doorName]);
}

function printConsoleOrNot(pText)
{
	if (typeof(console) === "object" && typeof(console.print) === "function")
		console.print(pText + "\r\n");
	else
		print(pText);
}
*/



/////////////////////////////////////////
// Functions

// Determines whether 2 strings are equal, ignoring case
function strsEqualIgnoringCase(pStr1, pStr2)
{
	// TODO: This still seems to be case-sensitive
	return pStr1.localeCompare(pStr2, undefined, { sensitivity: 'base' }) === 0;
}

// Returns whether a string is in an array of strings, ignoring case
function strInArrayIgnoringCase(pArrayOfStrs, pStr)
{
	var strFound = false;
	for (var i = 0; i < pArrayOfStrs.length && !strFound; ++i)
		strFound = strsEqualIgnoringCase(pStr, pArrayOfStrs[i]);
	return strFound;
}

// Parses a Synchronet node log file
//
// Parameters:
//  pLogFilename: The name of a Synchronet log file
//  pDoorNamesToSkip: Optional - An array of names of doors to not include
//  pUsernamesToSkip: Optional - An array of usernames to not include
//
// Return value: An array of objects containing the following properties:
//               loginDateTimeStr: The user's login date & time (string)
//               loginDate: A Date object representing the user's login date & time
//               nodeNum: The user's node number (number)
//               userNum: The user number (number)
//               username: The name of the user (string)
//               doorName: The name of the door (string)
function parseLogFile(pLogFilename, pDoorNamesToSkip, pUsernamesToSkip)
{
	var doorEntries = [];

	var inLogFile = new File(pLogFilename);
	if (inLogFile.open("r"))
	{
		var doorLogEntry = null;
		var lastTimeStr = "";
		var lastDateStr = "";
		var lastDate = null;
		var lastNodeNum = -1;
		var lastUserNum = -1;
		var lastUsername = "";
		var regexMatches = null;
		while (!inLogFile.eof)
		{
			// Read the next line from the log file
			var fileLine = inLogFile.readln(2048);

			// fileLine should be a string, but I've seen some cases
			// where for some reason it isn't.  If it's not a string,
			// then continue onto the next line.
			if (typeof(fileLine) != "string")
				continue;
			// If the line is blank, then skip it.
			if (fileLine.length == 0)
				continue;

			// Start of new log entry
			if (fileLine.match(/^@  [0-9][0-9]:[0-9][0-9].*[0-9][0-9][0-9][0-9]/) != null)
			{
				//          1         2         3         4
				//01234567890123456789012345678901234567890123456789
				//@  07:20p  Sun May 29 2022            Node   1
				lastTimeStr = fileLine.substr(3, 6);
				lastNodeNum = +(fileLine.substr(45));
				lastDateStr = fileLine.substr(11, 15);
				// Figure out the time and create a Date object for the last date.
				// The last character in the time string will be "a" for AM or "p" for PM.
				// If it's p, then add 12 to the hours.
				var lastTimeChar = lastTimeStr.charAt(lastTimeStr.length-1).toLowerCase();
				var hoursNum = +(lastTimeStr.substr(0, 2));
				if (lastTimeChar == "p" && hoursNum < 12)
					hoursNum += 12;
				var timeStr = format("%02d:", hoursNum) + lastTimeStr.substr(3, 2);
				lastDate = new Date(lastDateStr + " " + timeStr);
			}
			else if ((regexMatches = fileLine.match(/^\+\+ \(([0-9]+)\)  (.*).*Logon.*/)) != null)
			{
				// There should be 3 matches (the first is the whole line)
				if (regexMatches.length >= 3)
				{
					lastUserNum = +(regexMatches[1].trim());
					lastUsername = regexMatches[2].trim();
				}
			}
			else if (fileLine.indexOf("X- running external program: ") == 0)
			{
				var thisDoorName = fileLine.substr(29).trim();
				var addIt = true;
				if (pDoorNamesToSkip != null)
					addIt = !strInArrayIgnoringCase(pDoorNamesToSkip, thisDoorName);
				if (addIt && pUsernamesToSkip != null)
					addIt = !strInArrayIgnoringCase(pUsernamesToSkip, lastUsername);
				if (addIt)
				{
					doorEntries.push({
						loginDateTimeStr: lastDateStr + " " + lastTimeStr,
						loginDate: lastDate,
						nodeNum: lastNodeNum,
						userNum: lastUserNum,
						username: lastUsername,
						doorName: thisDoorName
					});
				}
			}
		}

		inLogFile.close();
	}

	return doorEntries;
}

// Compiles door stats from all available Synchronet node logs
//
// Parameters:
//  pDoorNamesToSkip: Optional - An array of names of doors to not include
//  pUsernamesToSkip: Optional - An array of usernames to not include
//  pForceParseAll: Optional boolean - Whether or not to parse all entries, regardless of an
//                  existing JSON file.  Defaults to false.
//  pVerbose: Optional boolean - Whether or not to enable verbose output. Defaults to false.
//
// Return value: An object containing door stats
function compileAllDoorStats(pDoorNamesToSkip, pUsernamesToSkip, pForceParseAll, pVerbose)
{
	var verboseOutput = (typeof(pVerbose) === "boolean" ? pVerbose : false);

	/*
	var stats = {
		startDate: null,
		endDate: null,
		byDoor: {},
		byUsername: {},
		doorsByPopularity: [],
		usersByMostDoorCount: []
	};
	*/

	// New
	var foundExistingStats = false;
	var forceParseAll = (typeof(pForceParseAll) === "boolean" ? pForceParseAll : false);
	var stats;
	if (forceParseAll)
		stats = getDefaultStatsObj();
	else
	{
		stats = readDoorStatsJSON(gStatsJsonFilename);
		if (Object.keys(stats).length == 0)
		{
			stats = getDefaultStatsObj();
			//Log(LOG_INFO, "syncLogDoorStats compileAllDoorStats(): Parsing all stats from the beginning");
		}
		else
		{
			foundExistingStats = true;
			//Log(LOG_INFO, "syncLogDoorStats compileAllDoorStats(): Read existing JSON file");
		}
	}
	// End New


	var logFilenames = directory(gLogsDir + "*.log");
	for (var logFileI = 0; logFileI < logFilenames.length; ++logFileI)
	{
		// New
		// If we read stats from an existing .json file, and the stats dates are valid, then
		// skip log files before the end date from the existing stats
		if (foundExistingStats && stats.startDate != null && stats.endDate != null)
		{
			// Log files are named by date (MMDDYY.log)
			var justFilename = file_getname(logFilenames[logFileI]);
			if (/^[0-9]{6}\.log$/.test(justFilename))
			{
				var yearNum = parseInt(justFilename.substr(4, 2));
				var monthNum = parseInt(justFilename.substr(0, 2));
				var dayNum = parseInt(justFilename.substr(2, 2));
				if (!isNaN(yearNum) && !isNaN(monthNum) && !isNaN(dayNum))
				{
					// Get the actual date from the file to get the full year
					var dateFromFile = getFirstDateFromLogFile(logFilenames[logFileI]);
					if (dateFromFile.year > 0)
					{
						var fileYearStr = dateFromFile.year.toString();
						if (fileYearStr.length > 2)
						{
							var actualYearStr = fileYearStr.substr(0, fileYearStr.length-2) + yearNum.toString();
							var tmpYearNum = parseInt(actualYearStr);
							if (!isNaN(tmpYearNum))
								yearNum = tmpYearNum;
						}
					}

					if (yearNum < stats.endDate.getFullYear())
						continue;
					else if (yearNum == stats.endDate.getFullYear())
					{
						var statsEndDateMonth = stats.endDate.getMonth()+1;
						if (monthNum < statsEndDateMonth)
							continue;
						else if (monthNum == statsEndDateMonth)
						{
							if (dayNum < stats.endDate.getDate())
								continue;
						}
					}
				}
			}
		}
		// End New

		var doorEntries = parseLogFile(logFilenames[logFileI], pDoorNamesToSkip, pUsernamesToSkip);
		for (var entryI = 0; entryI < doorEntries.length; ++entryI)
		{
			// New
			// If we read from a JSON file, then skip this log file if its login date is below the existing stats end date
			if (foundExistingStats && stats.startDate != null && doorEntries[entryI].loginDate < stats.endDate)
				continue;
			// End New

			if (stats.startDate == null)
				stats.startDate = doorEntries[entryI].loginDate;
			else
			{
				if (doorEntries[entryI].loginDate < stats.startDate)
					stats.startDate = doorEntries[entryI].loginDate;
			}
			if (stats.endDate == null)
				stats.endDate = doorEntries[entryI].loginDate;
			else
			{
				if (doorEntries[entryI].loginDate > stats.endDate)
					stats.endDate = doorEntries[entryI].loginDate;
			}

			// Update the by-door stats
			if (!stats.byDoor.hasOwnProperty(doorEntries[entryI].doorName))
			{
				stats.byDoor[doorEntries[entryI].doorName] = {
					numTimesRan: 0,
					lastRunLoginDateStr: "",
					lastRunUsername: "",
					usernameRunCounts: {}
				};
			}

			if (verboseOutput)
			{
				writeln("Updating stats for " + doorEntries[entryI].doorName);
				printObj(stats.byDoor[doorEntries[entryI].doorName]);
				writeln("");
			}

			stats.byDoor[doorEntries[entryI].doorName].numTimesRan += 1;
			stats.byDoor[doorEntries[entryI].doorName].lastRunLoginDateStr = doorEntries[entryI].loginDateTimeStr;
			stats.byDoor[doorEntries[entryI].doorName].lastRunLoginDate = doorEntries[entryI].loginDate;
			stats.byDoor[doorEntries[entryI].doorName].lastRunUsername = doorEntries[entryI].username;
			if (!stats.byDoor[doorEntries[entryI].doorName].usernameRunCounts.hasOwnProperty(doorEntries[entryI].username))
				stats.byDoor[doorEntries[entryI].doorName].usernameRunCounts[doorEntries[entryI].username] = 0;
			stats.byDoor[doorEntries[entryI].doorName].usernameRunCounts[doorEntries[entryI].username] += 1;

			// Update the by-username stats
			if (!stats.byUsername.hasOwnProperty(doorEntries[entryI].username))
			{
				stats.byUsername[doorEntries[entryI].username] = {
					totalNumDoorsRan: 0,
					lastLoginDateStr: "",
					doorRunCounts: {}
				};
			}
			stats.byUsername[doorEntries[entryI].username].totalNumDoorsRan += 1;
			stats.byUsername[doorEntries[entryI].username].lastLoginDateStr = doorEntries[entryI].loginDateTimeStr;
			stats.byUsername[doorEntries[entryI].username].lastLoginDate = doorEntries[entryI].loginDate;
			if (!stats.byUsername[doorEntries[entryI].username].doorRunCounts.hasOwnProperty(doorEntries[entryI].doorName))
				stats.byUsername[doorEntries[entryI].username].doorRunCounts[doorEntries[entryI].doorName] = 0;
			stats.byUsername[doorEntries[entryI].username].doorRunCounts[doorEntries[entryI].doorName] += 1;
		}
	}

	// Figure out the most common users for each door
	for (var doorName in stats.byDoor)
	{
		stats.byDoor[doorName].mostCommonUser = "";
		var doorRuns = 0;
		for (var username in stats.byDoor[doorName].usernameRunCounts)
		{
			if (stats.byDoor[doorName].usernameRunCounts[username] > doorRuns)
			{
				doorRuns = stats.byDoor[doorName].usernameRunCounts[username];
				stats.byDoor[doorName].mostCommonUser = username;
			}
		}
	}

	// Figure out the most-often run door per user
	for (var username in stats.byUsername)
	{
		stats.byUsername[username].mostCommonDoor = "";
		var doorRuns = 0;
		for (var doorName in stats.byUsername[username].doorRunCounts)
		{
			if (stats.byUsername[username].doorRunCounts[doorName] > doorRuns)
			{
				doorRuns = stats.byUsername[username].doorRunCounts[doorName];
				stats.byUsername[username].mostCommonDoor = doorName;
			}
		}
	}

	// Populate the stats.doorsByPopularity array
	for (var doorName in stats.byDoor)
	{
		// See if the doorsByPopularity array already has the door name, and if so, update that
		var dpIdx = -1;
		for (var dpI = 0; dpI < stats.doorsByPopularity.length && dpIdx == -1; ++dpI)
		{
			if (stats.doorsByPopularity[dpI].doorName == doorName)
				dpIdx = dpI;
		}
		if (dpIdx > -1)
			stats.doorsByPopularity[dpIdx].numTimesRan = stats.byDoor[doorName].numTimesRan;
		else
		{
			stats.doorsByPopularity.push({
				doorName: doorName,
				numTimesRan: stats.byDoor[doorName].numTimesRan
			});
		}
	}
	stats.doorsByPopularity.sort(function(doorA, doorB) {
		if (doorA.numTimesRan > doorB.numTimesRan)
			return -1;
		else if (doorA.numTimesRan < doorB.numTimesRan)
			return 1;
		else
			return 0;
	});

	// Populate the stats.usersByMostDoorCount array
	for (var username in stats.byUsername)
	{
		var totalNumDoorRuns = 0;
		for (var doorName in stats.byUsername[username].doorRunCounts)
			totalNumDoorRuns += stats.byUsername[username].doorRunCounts[doorName];
		// See if stats.usersByMostDoorCount already has the user. If so, update them.
		// Otherwise, add them.
		var userIdx = -1;
		for (var uIdx = 0; uIdx < stats.usersByMostDoorCount.length && userIdx == -1; ++uIdx)
		{
			if (stats.usersByMostDoorCount[uIdx].sername == username)
				userIdx = uIdx;
		}
		if (userIdx > -1)
		{
			stats.usersByMostDoorCount[userIdx].totalNumDoorsRan = stats.byUsername[username].totalNumDoorsRan;
			stats.usersByMostDoorCount[userIdx].mostCommonDoor = stats.byUsername[username].mostCommonDoor;
			stats.usersByMostDoorCount[userIdx].totalNumDoorRuns = totalNumDoorRuns;
			stats.usersByMostDoorCount[userIdx].doorRunCounts = stats.byUsername[username].doorRunCounts;
		}
		else
		{
			stats.usersByMostDoorCount.push({
				username: username,
				totalNumDoorsRan: stats.byUsername[username].totalNumDoorsRan,
				mostCommonDoor: stats.byUsername[username].mostCommonDoor,
				totalNumDoorRuns: totalNumDoorRuns,
				doorRunCounts: stats.byUsername[username].doorRunCounts
			});
		}
	}
	// Sort the users-by-most-door count array
	stats.usersByMostDoorCount.sort(function(userA, userB) {
		if (userA.totalNumDoorRuns > userB.totalNumDoorRuns)
			return -1;
		else if (userA.totalNumDoorRuns < userB.totalNumDoorRuns)
			return 1;
		else
			return 0;
	});
	// If read from a JSON file, sorting produces duplicates for some reason, so eliminate
	// duplicates in that case.
	if (foundExistingStats)
	{
		var seenUsernames = {};
		var newUsersByMostDoorCount = [];
		for (var i = 0; i < stats.usersByMostDoorCount.length; ++i)
		{
			if (!seenUsernames.hasOwnProperty(stats.usersByMostDoorCount[i].username))
				newUsersByMostDoorCount.push(stats.usersByMostDoorCount[i]);
			seenUsernames[stats.usersByMostDoorCount[i].username] = true;
		}
		stats.usersByMostDoorCount = newUsersByMostDoorCount;
	}
	

	return stats;
}

// Gets the first date from a Synchroent node log file
//
// Parameters:
//  pFilename: The name of a Synchronet log file to read
//
// Return value: An object with the following properties:
//               year: The year (0 if not found)
//               month: The month number, 1-12 (0 if not found)
//               day: The day of the month (0 if not found)
function getFirstDateFromLogFile(pFilename)
{
	var retObj = {
		year: 0,
		month: 0,
		day: 0
	};

	var inFile = new File(pFilename);
	if (inFile.open("r"))
	{
		while (!inFile.eof && retObj.year == 0 && retObj.month == 0 && retObj.day == 0)
		{
			// Read the next line from the log file
			var fileLine = inFile.readln(2048);
			// fileLine should be a string, but I've seen some cases
			// where for some reason it isn't.  If it's not a string,
			// then continue onto the next line.
			if (typeof(fileLine) != "string")
				continue;
			// If the line is blank, then skip it.
			if (fileLine.length == 0)
				continue;

			// Start of new log entry
			var regexMatches = null;
			if ((regexMatches = fileLine.match(/^@  [0-9][0-9]:[0-9][0-9].*[a-zA-Z]{3} ([a-zA-Z]{3}) ([0-9]{2}) ([0-9]+)\s+Node\s+[0-9]/)) != null)
			{
				if (regexMatches.length >= 4)
				{
					var yearNum = parseInt(regexMatches[3]);
					if (!isNaN(yearNum))
						retObj.year = yearNum;
					var dayNum = parseInt(regexMatches[2]);
					if (!isNaN(dayNum))
						retObj.day = dayNum;
					if (regexMatches[1] == "Jan")
						retObj.month = 1;
					else if (regexMatches[1] == "Feb")
						retObj.month = 2;
					else if (regexMatches[1] == "Feb")
						retObj.month = 2;
					else if (regexMatches[1] == "Mar")
						retObj.month = 3;
					else if (regexMatches[1] == "Apr")
						retObj.month = 4;
					else if (regexMatches[1] == "May")
						retObj.month = 5;
					else if (regexMatches[1] == "Jun")
						retObj.month = 6;
					else if (regexMatches[1] == "Jul")
						retObj.month = 7;
					else if (regexMatches[1] == "Aug")
						retObj.month = 8;
					else if (regexMatches[1] == "Sep")
						retObj.month = 9;
					else if (regexMatches[1] == "Oct")
						retObj.month = 10;
					else if (regexMatches[1] == "Nov")
						retObj.month = 11;
					else if (regexMatches[1] == "Dec")
						retObj.month = 12;
					break;
				}
			}
		}

		inFile.close();
	}
	return retObj;
}

// Returns a default top-level door stats object, for use with compileAllDoorStats()
function getDefaultStatsObj()
{
	return {
		startDate: null,
		endDate: null,
		byDoor: {},
		byUsername: {},
		doorsByPopularity: [],
		usersByMostDoorCount: []
	};
}

// Reads a JSON file containing the door stats. Should be the same as
// what was generated by compileAllDoorStats().
//
// Parameters:
//  pJSONFilename: The name of the JSON file to read
//
// Return value: An object with the door stats (should be the same as what was
//               generated by compileAllDoorStats())
function readDoorStatsJSON(pJSONFilename)
{
	var doorStatsObj;
	if (file_exists(pJSONFilename))
	{
		var statsJsonFile = new File(pJSONFilename);
		if (statsJsonFile.open("r"))
		{
			var statsFileContents = statsJsonFile.read(statsJsonFile.length);
			statsJsonFile.close();
			try
			{
				doorStatsObj = JSON.parse(statsFileContents);
				if (typeof(doorStatsObj.startDate) === "string")
					doorStatsObj.startDate = new Date(doorStatsObj.startDate);
				if (typeof(doorStatsObj.endDate) === "string")
					doorStatsObj.endDate = new Date(doorStatsObj.endDate);
			}
			catch (error)
			{
				log(LOG_ERR, "Error parsing door stats JSON: " + error);
			}
		}
	}
	else
		doorStatsObj = {};
	return doorStatsObj;
}

// Prints an object, for debugging purposes
function printObj(pObj, pNumSpaces)
{
	var numSpaces = (typeof(pNumSpaces) === "number" && pNumSpaces > 0 ? pNumSpaces : 0);
	var indentationStr = "";
	for (var i = 0; i < numSpaces; ++i)
		indentationStr += " ";

	for (var prop in pObj)
	{
		if (typeof(pObj[prop]) === "object")
		{
			writeln(indentationStr + " " + prop + ":");
			printObj(pObj[prop], numSpaces + 1);
		}
		else
			writeln(indentationStr + " " + prop + ": " + pObj[prop]);
	}
}
